"use strict";

const fs = require("fs-extra");
const { checkFileExistsBeforeUpload } = require("./upload-file");
const { downloadFile } = require("./download");
const { autoGenerateSlug } = require("./utils");

// This script is used to seed content types in the Strapi project.
// It is automatically generated by the XSite CLI. Do not modify it.

// Uploads files within a data object, checking if they exist before uploading
async function uploadDataFilesWithinDataObject(data) {
  if (typeof data !== "object" || data === null) return data;

  if (
    data.url &&
    data.alt &&
    typeof data.url === "string" &&
    data.url.startsWith("/")
  ) {
    try {
      return await checkFileExistsBeforeUpload([data.url], data?.alt);
    } catch (err) {
      console.error(`Error uploading local file ${data.url}:`, err);
      return null;
    }
  }

  const matchers = ["http:", "https:", "www"];
  if (
    data.url &&
    data.alt &&
    typeof data.url === "string" &&
    data.url.split("/").some((item) => matchers.includes(item))
  ) {
    try {
      const filename = await downloadFile(data);
      return await checkFileExistsBeforeUpload([filename], data?.alt || "alt");
    } catch (err) {
      console.error(`[ERROR] Error downloading file ${data.url}:`, err);
      return null;
    }
  }

  if (Array.isArray(data)) {
    return await Promise.all(data.map(uploadDataFilesWithinDataObject));
  }

  const processed = {};
  for (const key in data) {
    if (data.hasOwnProperty(key)) {
      processed[key] = await uploadDataFilesWithinDataObject(data[key]);
    }
  }
  return processed;
}

async function createRelatedEntries(uid, data) {
  const schema = strapi.contentTypes[uid] || strapi.components[uid];
  if (!schema) return data;

  const processed = {};

  for (const key in data) {
    const value = data[key];
    const field = schema.attributes?.[key];

    // If the field is not defined in the schema or the value is null, keep it as is
    if (!field || value == null) {
      processed[key] = value;
      continue;
    }

    if (field.type === "relation") {
      const targetUid = field.target;

      // If it's an array (many relations), process each one
      if (Array.isArray(value)) {
        const createdEntries = [];
        for (const item of value) {
          const targetSchema = strapi.contentTypes[targetUid];
          if (targetSchema) {
            autoGenerateSlug(item, targetSchema);
          }

          const cleanedItem = await createRelatedEntries(targetUid, item);
          const created = await strapi.documents(targetUid).create({
            data: cleanedItem,
          });
          await strapi.documents(targetUid).publish({
            documentId: created.documentId,
          });
          createdEntries.push(created);
        }
        processed[key] = createdEntries;
      } else {
        // For single relation, process and create just one related entry
        const targetSchema = strapi.contentTypes[targetUid];
        if (targetSchema) {
          autoGenerateSlug(value, targetSchema);
        }

        const cleaned = await createRelatedEntries(targetUid, value);
        const created = await strapi.documents(targetUid).create({
          data: cleaned,
        });
        await strapi.documents(targetUid).publish({
          documentId: created.documentId,
        });
        processed[key] = created;
      }
    } else if (field.type === "component") {
      // Handle component fields
      const componentUid = field.component;
      if (field.repeatable) {
        processed[key] = await Promise.all(
          value.map((item) => createRelatedEntries(componentUid, item))
        );
      } else {
        processed[key] = await createRelatedEntries(componentUid, value);
      }
    } else if (field.type === "dynamiczone") {
      // Handle dynamic zones fields
      processed[key] = await Promise.all(
        value.map(async (block) => {
          const { __component, ...rest } = block;
          const resolvedComponent = await createRelatedEntries(
            __component,
            rest
          );
          return { __component, ...resolvedComponent };
        })
      );
    } else if (field.type === "media") {
      processed[key] = await uploadDataFilesWithinDataObject(value);
    } else {
      processed[key] = value;
    }
  }

  return processed;
}

async function resolveRelationsForConnection(uid, data) {
  const schema = strapi.contentTypes[uid] || strapi.components[uid];
  if (!schema) return data;

  const resolved = {};

  for (const key in data) {
    const value = data[key];

    const field = schema.attributes?.[key];

    if (!field || value == null) {
      resolved[key] = value;
      continue;
    }

    if (field.type === "relation") {
      if (Array.isArray(value)) {
        resolved[key] = value.map((entry) => {
          if (entry.isFeatured === true || entry.featured === true) {
            return entry.id;
          }
        });
      } else {
        resolved[key] = value.id;
      }
    } else if (field.type === "component") {
      const componentUid = field.component;
      if (field.repeatable) {
        resolved[key] = await Promise.all(
          value.map((item) => resolveRelationsForConnection(componentUid, item))
        );
      } else {
        resolved[key] = await resolveRelationsForConnection(
          componentUid,
          value
        );
      }
    } else if (field.type === "dynamiczone") {
      resolved[key] = await Promise.all(
        value.map(async (block) => {
          const { __component, ...rest } = block;
          const resolvedComponent = await resolveRelationsForConnection(
            __component,
            rest
          );
          return { __component, ...resolvedComponent };
        })
      );
    } else {
      resolved[key] = value;
    }
  }

  return resolved;
}

async function createEntry({ model, entry }) {
  try {
    const uid = `api::${model}.${model}`;
    const schema = strapi.contentTypes[uid];

    if (schema) {
      entry = autoGenerateSlug(entry, schema);
    }

    const dataWithCreatedRelations = await createRelatedEntries(uid, entry);
    const resolvedData = await resolveRelationsForConnection(
      uid,
      dataWithCreatedRelations
    );

    const isSingleType = schema?.kind === "singleType";

    // For single-types, delete existing entries before creating a new one
    if (isSingleType) {
      const existing = await strapi.documents(uid).findMany();
      for (const existingEntry of existing) {
        await strapi.documents(uid).delete({
          documentId: existingEntry.documentId,
        });
      }
    }

    // For collection-types, if an entry with the same slug exists, update it instead of creating a duplicate
    if (!isSingleType && entry.slug) {
      const existingPages = await strapi.query(uid).findMany({
        filters: { slug: entry.slug },
      });
      if (existingPages.length) {
        await strapi.documents(uid).update({
          documentId: existingPages[0].documentId,
          data: resolvedData,
        });
        await strapi.documents(uid).publish({
          documentId: existingPages[0].documentId,
        });
        return;
      }
    }

    // For non-existing entries, create a new one and publish it
    const data = await strapi.documents(uid).create({
      data: {
        ...resolvedData,
        publishedAt: new Date(),
      },
    });

    await strapi.documents(uid).publish({
      documentId: data.documentId,
    });
  } catch (error) {
    console.error(`[ERROR] Creating ${model} entry:`, error);
    throw error;
  }
}

async function seedData() {
  const data = await fs.readJson("./data/data.json");
  const processedData = await uploadDataFilesWithinDataObject(data);

  const contentTypes = Object.keys(processedData);

  console.log(
    `[INFO] Seeding data for ${contentTypes.length} content types...`
  );

  // Processing each content type (Collection-type or Single-type)
  for (const contentType of contentTypes) {
    const contentData = processedData[contentType];
    console.log(`[INFO] Processing ${contentType} entries...`);

    // Handling Collection-type entries
    if (Array.isArray(contentData)) {
      for (const entry of contentData) {
        await createEntry({
          model: contentType,
          entry,
        });
      }
    } else if (contentData && typeof contentData === "object") {
      // Handling Single-type entry
      await createEntry({
        model: contentType,
        entry: contentData,
      });
    }
  }
  console.log(`[INFO] Data seeding completed successfully.`);
}

module.exports = {
  seedData,
};
