const { SyntaxKind, VariableDeclarationKind, Project } = require("ts-morph");
const path = require("path");
const xsite = require("../../xsite.json");
require("dotenv").config({ path: path.resolve(__dirname, "../.env") });

const project = new Project({
  tsConfigFilePath: "./tsconfig.json",
  skipAddingFilesFromTsConfig: false,
});

const formatFontConfig = (fontDetail: string): string => {
  return fontDetail.replace(/"(\w+)":/g, "$1:");
};

const fetchFontDetail = async (fontname: string): Promise<string> => {
  try {
    const url = "https://xsite.live/registry/font.json";
    const response = await fetch(url, {
      method: "GET",
      headers: { "Content-Type": "application/json" },
    });

    if (!response.ok) {
      throw new Error("Failed to fetch font metadata");
    }

    const responseData = await response.json();
    return responseData[fontname];
  } catch (error: any) {
    throw new Error(`Font fetch error: ${error.message}`);
  }
};

const handleFontImport = (layoutFile: any, fontName: string): void => {
  const existingImport = layoutFile.getImportDeclaration("next/font/google");

  if (!existingImport) {
    const allImports = layoutFile.getImportDeclarations();
    const lastImport = allImports[allImports.length - 1];

    layoutFile.insertImportDeclaration(lastImport.getChildIndex() + 1, {
      moduleSpecifier: "next/font/google",
      namedImports: [fontName],
    });
    console.log(`[INFO] Font import added: ${fontName}`);
  } else {
    const existingNamedImports = existingImport.getNamedImports();
    const fontAlreadyImported = existingNamedImports.some(
      (imp: any) => imp.getName() === fontName
    );

    if (!fontAlreadyImported) {
      existingImport.addNamedImport(fontName);
      console.log(`[INFO] Font import updated: ${fontName}`);
    } else {
      console.log(`[INFO] Font import already exists: ${fontName}`);
    }
  }
};

const handleFontVariable = (
  layoutFile: any,
  fontName: string,
  fontDetail: string
): void => {
  const lowerCaseFontName = fontName.toLowerCase();
  const existingFont = layoutFile.getVariableDeclaration(lowerCaseFontName);

  const formattedFontDetail = formatFontConfig(fontDetail);

  if (!existingFont) {
    const allImports = layoutFile.getImportDeclarations();
    const insertIndex = allImports[allImports.length - 1].getChildIndex() + 2;

    layoutFile.insertVariableStatement(insertIndex, {
      declarationKind: VariableDeclarationKind.Const,
      declarations: [
        {
          name: lowerCaseFontName,
          initializer: formattedFontDetail,
        },
      ],
    });
    console.log(`[SUCCESS] Font variable added: ${lowerCaseFontName}`);
  } else {
    const variableStatement = existingFont.getVariableStatement();
    const insertIndex = variableStatement.getChildIndex();

    variableStatement.remove();
    layoutFile.insertVariableStatement(insertIndex, {
      declarationKind: VariableDeclarationKind.Const,
      declarations: [
        {
          name: lowerCaseFontName,
          initializer: formattedFontDetail,
        },
      ],
    });
    console.log(`[WARN] Font variable updated: ${lowerCaseFontName}`);
  }
};

const handleBodyClassName = (layoutFile: any, fontName: string): void => {
  const lowerCaseFontName = fontName.toLowerCase();
  const bodyElement = layoutFile
    .getDescendantsOfKind(SyntaxKind.JsxElement)
    .find(
      (el: any) => el.getOpeningElement().getTagNameNode().getText() === "body"
    );

  if (!bodyElement) {
    console.log("[ERROR] Body element not found");
    return;
  }

  const opening = bodyElement.getOpeningElement();
  const classAttr = opening.getAttribute("className");
  const expectedClassName = `{${lowerCaseFontName}.className}`;

  if (!classAttr) {
    opening.addAttribute({
      name: "className",
      initializer: expectedClassName,
    });
    console.log("[SUCCESS] Body className added");
  } else {
    const currentValue = classAttr.getInitializer()?.getText();
    if (currentValue !== expectedClassName) {
      classAttr.setInitializer(expectedClassName);
      console.log("[WARN] Body className updated");
    } else {
      console.log("[INFO] Body className already correct");
    }
  }
};

const loadFont = async (): Promise<void> => {
  const fontName = xsite.fontname;
  console.log(`[INFO] Processing font: ${fontName}`);

  const layoutFilePath = "app/layout.tsx";
  const layoutFile = project.getSourceFile(layoutFilePath);

  if (!layoutFile) {
    console.log("[ERROR] Layout file not found");
    process.exit(1);
  }

  const fontDetail = await fetchFontDetail(fontName);
  console.log("[INFO] Font metadata fetched successfully");

  handleFontImport(layoutFile, fontName);
  handleFontVariable(layoutFile, fontName, fontDetail);
  handleBodyClassName(layoutFile, fontName);

  layoutFile.formatText();
  await project.save();

  console.log(`[SUCCESS] Font ${fontName} injection completed`);
};

const main = async (): Promise<void> => {
  try {
    if (!xsite.fontname) {
      throw new Error("xsite configuration missing fontname");
    }

    console.log("[INFO] Starting font injection process...");
    await loadFont();
  } catch (error: any) {
    console.log(`[ERROR] ${error.message}`);
    if (error.stack) {
      console.log(`[DEBUG] Stack trace: ${error.stack}`);
    }
    process.exit(1);
  }
};

main();

