"use strict";

const fs = require("fs-extra");
const { checkFileExistsBeforeUpload } = require("./upload-file");
const { downloadFile } = require("./download");
const { autoGenerateSlug } = require("./utils");
const { handleNavigationEntry } = require("./navigation");
const { handleAudienceEntry } = require("./audience");

// This script is used to seed content types in the Strapi project.
// It is automatically generated by the XSite CLI. Do not modify it.

// Uploads files within a data object, checking if they exist before uploading
async function uploadDataFilesWithinDataObject(data) {
  if (typeof data !== "object" || data === null) return data;

  if (
    data.hasOwnProperty("url") &&
    data.hasOwnProperty("alternativeText") &&
    typeof data.url === "string" &&
    data.url.startsWith("/")
  ) {
    try {
      return await checkFileExistsBeforeUpload(
        [data.url],
        data?.alternativeText || "alt"
      );
    } catch (err) {
      console.error(`Error uploading local file ${data.url}:`, err);
      return null;
    }
  }

  const matchers = ["http:", "https:", "www"];
  if (
    data.hasOwnProperty("url") &&
    data.hasOwnProperty("alternativeText") &&
    typeof data.url === "string" &&
    data.url.split("/").some((item) => matchers.includes(item))
  ) {
    try {
      const filename = await downloadFile(data);
      return await checkFileExistsBeforeUpload(
        [filename],
        data?.alternativeText || "alt"
      );
    } catch (err) {
      console.error(`[ERROR] Error downloading file ${data.url}:`, err);
      return null;
    }
  }

  if (Array.isArray(data)) {
    return await Promise.all(data.map(uploadDataFilesWithinDataObject));
  }

  const processed = {};
  for (const key in data) {
    if (data.hasOwnProperty(key)) {
      processed[key] = await uploadDataFilesWithinDataObject(data[key]);
    }
  }
  return processed;
}

async function createRelatedEntries(app, uid, data) {
  const schema = app.contentTypes[uid] || app.components[uid];
  if (!schema) return data;

  const processed = {};

  for (const key in data) {
    const value = data[key];
    const field = schema.attributes?.[key];

    // If the field is not defined in the schema or the value is null, keep it as is
    if (!field || value == null) {
      processed[key] = value;
      continue;
    }

    if (field.type === "relation") {
      const targetUid = field.target;

      // If it's an array (many relations), process each one
      if (Array.isArray(value)) {
        const createdEntries = [];
        for (const item of value) {
          const targetSchema = app.contentTypes[targetUid];
          if (targetSchema) {
            autoGenerateSlug(item, targetSchema);
          }

          const cleanedItem = await createRelatedEntries(app, targetUid, item);
          const created = await app.documents(targetUid).create({
            data: cleanedItem,
          });
          await app.documents(targetUid).publish({
            documentId: created.documentId,
          });
          createdEntries.push(created);
        }
        processed[key] = createdEntries;
      } else {
        // For single relation, process and create just one related entry
        const targetSchema = app.contentTypes[targetUid];
        if (targetSchema) {
          autoGenerateSlug(value, targetSchema);
        }

        const cleaned = await createRelatedEntries(app, targetUid, value);
        const created = await app.documents(targetUid).create({
          data: cleaned,
        });
        await app.documents(targetUid).publish({
          documentId: created.documentId,
        });
        processed[key] = created;
      }
    } else if (field.type === "component") {
      // Handle component fields
      const componentUid = field.component;
      if (field.repeatable) {
        processed[key] = await Promise.all(
          value.map((item) => createRelatedEntries(app, componentUid, item))
        );
      } else {
        processed[key] = await createRelatedEntries(app, componentUid, value);
      }
    } else if (field.type === "dynamiczone") {
      // Handle dynamic zones fields
      processed[key] = await Promise.all(
        value.map(async (block) => {
          const { __component, ...rest } = block;
          const resolvedComponent = await createRelatedEntries(
            app,
            __component,
            rest
          );
          return { __component, ...resolvedComponent };
        })
      );
    } else if (field.type === "media") {
      processed[key] = await uploadDataFilesWithinDataObject(value);
    } else {
      processed[key] = value;
    }
  }

  return processed;
}

async function resolveRelationsForConnection(app, uid, data) {
  const schema = app.contentTypes[uid] || app.components[uid];
  if (!schema) return data;

  const resolved = {};

  for (const key in data) {
    const value = data[key];

    const field = schema.attributes?.[key];

    if (!field || value == null) {
      resolved[key] = value;
      continue;
    }

    if (field.type === "relation") {
      if (Array.isArray(value)) {
        resolved[key] = value.map((entry) => {
          if (entry.isFeatured === true || entry.featured === true) {
            return entry.id;
          }
        });
      } else {
        resolved[key] = value.id;
      }
    } else if (field.type === "component") {
      const componentUid = field.component;
      if (field.repeatable) {
        resolved[key] = await Promise.all(
          value.map((item) =>
            resolveRelationsForConnection(app, componentUid, item)
          )
        );
      } else {
        resolved[key] = await resolveRelationsForConnection(
          app,
          componentUid,
          value
        );
      }
    } else if (field.type === "dynamiczone") {
      resolved[key] = await Promise.all(
        value.map(async (block) => {
          const { __component, ...rest } = block;
          const resolvedComponent = await resolveRelationsForConnection(
            app,
            __component,
            rest
          );
          return { __component, ...resolvedComponent };
        })
      );
    } else {
      resolved[key] = value;
    }
  }

  return resolved;
}

async function createEntry({ app, model, entry }) {
  try {
    const uid = `api::${model}.${model}`;
    const schema = app.contentTypes[uid];

    if (schema) {
      entry = autoGenerateSlug(entry, schema);
    }

    const dataWithCreatedRelations = await createRelatedEntries(
      app,
      uid,
      entry
    );
    const resolvedData = await resolveRelationsForConnection(
      app,
      uid,
      dataWithCreatedRelations
    );

    const isSingleType = schema?.kind === "singleType";

    // For single-types, delete existing entries before creating a new one
    if (isSingleType) {
      const existing = await app.documents(uid).findMany();
      for (const existingEntry of existing) {
        await app.documents(uid).delete({
          documentId: existingEntry.documentId,
        });
      }
    }

    // For collection-types, if an entry with the same slug exists, update it instead of creating a duplicate
    if (!isSingleType && entry.slug) {
      const existingPages = await app.query(uid).findMany({
        filters: { slug: entry.slug },
      });
      if (existingPages.length) {
        await app.documents(uid).update({
          documentId: existingPages[0].documentId,
          data: resolvedData,
        });
        await app.documents(uid).publish({
          documentId: existingPages[0].documentId,
        });
        return;
      }
    }

    const data = await app.documents(uid).create({
      data: {
        ...resolvedData,
        publishedAt: new Date(),
      },
    });

    await app.documents(uid).publish({
      documentId: data.documentId,
    });
  } catch (error) {
    app.log.error(`Creating ${model} entry:`, error);
    throw error;
  }
}

async function seedContentTypes(app) {
  try {
    const data = await fs.readJson("./data/data.json");
    const processedData = await data;

    const {
      navigation: navigationData,
      audience: audienceData,
      ...otherContentTypes
    } = processedData;

    const contentTypes = Object.keys(otherContentTypes);

    await handleAudienceEntry(app, audienceData);
    await handleNavigationEntry(app, navigationData);

    app.log.info(` Seeding data for ${contentTypes.length} content types...`);

    // Processing each content type (Collection-type or Single-type)
    for (const contentType of contentTypes) {
      const contentData = processedData[contentType];
      app.log.info(` Processing ${contentType} entries...`);

      // Handling Collection-type entries
      if (Array.isArray(contentData)) {
        for (const entry of contentData) {
          await createEntry({
            app,
            model: contentType,
            entry,
          });
        }
      } else if (contentData && typeof contentData === "object") {
        // Handling Single-type entry
        await createEntry({
          app,
          model: contentType,
          entry: contentData,
        });
      }
    }
    app.log.info("Seeding data completed.");
  } catch (error) {
    app.log.error(`Seeding data failed:`, error);
  }
}

module.exports = {
  seedContentTypes,
};
