const fs = require("fs");
const path = require("path");
const axios = require("axios");
const mime = require("mime-types");
const { checkFileExistsBeforeUpload } = require("./upload-file");
const xsite = require("../../xsite.json");

const DOWNLOAD_FOLDER = path.join(path.resolve(), "data", "uploads");

async function fetchProjectAssets() {
  const { data } = await axios.get(
    `${xsite.url}/project/${xsite.projectId}/get-all-assets`,
    {
      headers: {
        "xsite-api-key": xsite["xsite-api-key"],
        "xsite-api-secret": xsite["xsite-api-secret"],
      },
    }
  );
  return data.data;
}

async function generateFilename(file) {
  let filename = "unnamed_file";
  try {
    const parsed = new URL(file.url);
    const base = path.basename(parsed.pathname);

    if (path.extname(base)) {
      filename = base;
    } else {
      try {
        const head = await axios.head(file.url);
        const mimeType = head.headers["content-type"];
        const ext = mime.extension(mimeType) || "jpg";
        filename = `${base || "file"}.${ext}`;
      } catch {
        filename = `${base || "file"}.jpg`;
      }
    }
  } catch {
    filename = "fallback.jpg";
  }
  return filename;
}

async function downloadFileFromUrl(url, filePath) {
  const response = await axios({
    method: "GET",
    url,
    responseType: "stream",
  });

  const writer = fs.createWriteStream(filePath);
  response.data.pipe(writer);

  return new Promise((resolve, reject) => {
    writer.on("finish", resolve);
    writer.on("error", reject);
  });
}

async function downloadFile(file) {
  try {
    const filename = await generateFilename(file);
    const filePath = path.join(DOWNLOAD_FOLDER, filename);

    if (!fs.existsSync(filePath)) {
      await downloadFileFromUrl(file.url, filePath);
    }

    return await checkFileExistsBeforeUpload([filename], file?.alternativeText);
  } catch (error) {
    console.warn(`[WARN] Failed to process file ${file.url}:`, error.message);
    return null;
  }
}

async function downloadAllProjectFiles(app) {
  try {
    if (!fs.existsSync(DOWNLOAD_FOLDER)) {
      fs.mkdirSync(DOWNLOAD_FOLDER, { recursive: true });
    }

    const files = await fetchProjectAssets();
    if (!files?.assets?.length) {
      console.log("[INFO] No files to download");
      return;
    }

    console.log(`[INFO] Starting download of ${files.assets.length} files...`);

    for (const file of files.assets) {
      await downloadFile(file);
    }

    console.log("[INFO] All downloads complete");
  } catch (error) {
    console.error(`[ERROR] Download process failed: ${error.message}`);
    throw error;
  }
}

module.exports = {
  downloadAllProjectFiles,
  downloadFile,
};
