"use strict";

const { createStrapi, compileStrapi } = require("@strapi/strapi");
const {
  setPublicPermissions,
  setNavigationPluginConfiguration,
} = require("./configurations");
const { fetchAndSaveProjectJSONDataTOFile } = require("./utils");
const { validateXSiteConfig } = require("./utils");
const { seedContentTypes } = require("./content-types");
const { downloadAllProjectFiles } = require("./download");

async function main() {
  try {
    const appContext = await compileStrapi();
    const app = await createStrapi(appContext).load();
    app.log.info("Strapi initialized");

    app.log.info("Configuring navigation plugin...");
    await setNavigationPluginConfiguration(app);
    app.log.info("Navigation plugin configured successfully");

    app.log.info("Setting public permissions...");
    await setPublicPermissions(app);
    app.log.info("Public permissions set successfully");

    app.log.info("Validating XSite configuration...");
    validateXSiteConfig();
    app.log.info("XSite configuration validated successfully");

    app.log.info("Starting data fetching process...");
    await fetchAndSaveProjectJSONDataTOFile(app);
    app.log.info("Data fetching process completed successfully.");

    app.log.info("Downloading all project files...");
    await downloadAllProjectFiles(app);
    app.log.info("All project files downloaded successfully.");

    app.log.info("Seeding content types...");
    await seedContentTypes(app);
    app.log.info("Content types seeded successfully.");

    app.log.info("Strapi seeding completed successfully.");
    await app.destroy();
    process.exit(0);
  } catch (error) {
    app.log.error("Error seeding Strapi:", error);
    process.exit(1);
  }
}

main().catch((error) => {
  console.error("Unhandled error:", error);
  process.exit(1);
});
