async function handleNavigationEntry(app, entries) {
  const NAV_UID = "plugin::navigation.navigation";
  const ITEM_UID = "plugin::navigation.navigation-item";
  const AUD_UID = "plugin::navigation.audience";

  const now = () => new Date();

 // Remove default navigation if exists
  const [defaultNav] = await app.documents(NAV_UID).findMany({
    where: { slug: "navigation" },
  });

  if (defaultNav) {
    await app.documents(NAV_UID).delete({ documentId: defaultNav.documentId });
  }

  const normalizeItem = (raw) => ({
    title: raw.title,
    type: raw.type,
    path: raw.type === "EXTERNAL" ? null : raw.path ?? raw.internalPath ?? null,
    externalPath: raw.type === "EXTERNAL" ? raw.path ?? null : null,
    uiRouterKey: raw.uiRouterKey ?? raw.path ?? raw.internalPath ?? null,
    order: Number.isInteger(raw.order) ? raw.order : 0,
    menuAttached:
      typeof raw.menuAttached === "boolean" ? raw.menuAttached : true,
    publishedAt: now(),
  });

  const audienceCache = { byKey: new Map(), byName: new Map() };

  const getAudienceDocIds = async (rawItem) => {
    if (
      Array.isArray(rawItem?.audienceDocIds) &&
      rawItem.audienceDocIds.length
    ) {
      return rawItem.audienceDocIds.filter(Boolean);
    }

    const fromDocs = (rawItem?.audiences || [])
      .map((a) => a?.documentId)
      .filter(Boolean);
    if (fromDocs.length) return fromDocs;

    const wanted = (rawItem?.audiences || [])
      .map((a) => ({ key: a?.key, name: a?.name }))
      .filter((a) => a.key || a.name);

    const docIds = [];
    for (const a of wanted) {
      if (a.key) {
        if (!audienceCache.byKey.has(a.key)) {
          const [hit] = await app
            .documents(AUD_UID)
            .findMany({ where: { key: a.key }, limit: 1 });
          audienceCache.byKey.set(a.key, hit?.documentId || null);
        }
        const id = audienceCache.byKey.get(a.key);
        if (id) docIds.push(id);
      } else if (a.name) {
        if (!audienceCache.byName.has(a.name)) {
          const [hit] = await app
            .documents(AUD_UID)
            .findMany({ where: { name: a.name }, limit: 1 });
          audienceCache.byName.set(a.name, hit?.documentId || null);
        }
        const id = audienceCache.byName.get(a.name);
        if (id) docIds.push(id);
      }
    }
    return docIds;
  };

  const getOrCreateNavigation = async (entry) => {
    const [existing] = await app.documents(NAV_UID).findMany({
      where: { slug: entry.slug },
      limit: 1,
    });

    if (existing) {
      await app.documents(NAV_UID).update({
        documentId: existing.documentId,
        data: {
          name: entry.name,
          slug: entry.slug,
          visible: !!entry.visible,
          publishedAt: now(),
        },
      });
      return existing.documentId;
    }

    const created = await app.documents(NAV_UID).create({
      data: {
        name: entry.name,
        slug: entry.slug,
        visible: !!entry.visible,
        publishedAt: now(),
      },
    });
    return created.documentId;
  };

  const upsertTopLevelItem = async (navigationDocId, rawItem) => {
    const base = normalizeItem(rawItem);

    const where = { master: { documentId: navigationDocId }, parent: null };
    if (base.uiRouterKey) where.uiRouterKey = base.uiRouterKey;
    else if (base.path) where.path = base.path;
    else if (base.externalPath) where.externalPath = base.externalPath;
    else where.title = base.title;

    const [existing] = await app
      .documents(ITEM_UID)
      .findMany({ where, limit: 1 });
    const data = {
      ...base,
      master: { set: [{ documentId: navigationDocId }] },
    };

    if (existing) {
      const res = await app
        .documents(ITEM_UID)
        .update({ documentId: existing.documentId, data });
      return res.documentId;
    } else {
      const created = await app.documents(ITEM_UID).create({ data });
      return created.documentId;
    }
  };

  for (const entry of entries) {
    if (!entry?.name || !entry?.slug) continue;

    const navigationDocId = await getOrCreateNavigation(entry);
    const rootItems = Array.isArray(entry.items) ? entry.items : [];

    const allItems = [];
    const stack = [...rootItems];
    while (stack.length) {
      const it = stack.shift();
      allItems.push(it);
      if (Array.isArray(it.items) && it.items.length)
        stack.unshift(...it.items);
    }

    const bySourceId = new Map();
    const byPath = new Map();

    for (const it of allItems) {
      if (!it?.title || !it?.type) continue;
      const docId = await upsertTopLevelItem(navigationDocId, it);
      if (it.id) bySourceId.set(it.id, docId);
      if (it.path) byPath.set(it.path, docId);
    }

    const linkRecursively = async (list, inheritedParentDocId = null) => {
      for (const it of list || []) {
        const childDocId =
          (it.id && bySourceId.get(it.id)) ||
          (it.path && byPath.get(it.path)) ||
          null;

        let parentDocId = inheritedParentDocId;
        if (!parentDocId && it.parentId)
          parentDocId = bySourceId.get(it.parentId) || null;
        if (!parentDocId && it.parentPath)
          parentDocId = byPath.get(it.parentPath) || null;

        if (childDocId) {
          const audienceDocIds = await getAudienceDocIds(it);
          const patch = {
            master: { set: [{ documentId: navigationDocId }] },
            publishedAt: now(),
            ...(parentDocId
              ? { parent: { set: [{ documentId: parentDocId }] } }
              : {}),
            ...(audienceDocIds?.length
              ? {
                  audience: {
                    set: audienceDocIds.map((d) => ({ documentId: d })),
                  },
                }
              : {}),
          };
          await app
            .documents(ITEM_UID)
            .update({ documentId: childDocId, data: patch });
        }

        if (Array.isArray(it.items) && it.items.length) {
          await linkRecursively(it.items, childDocId || parentDocId);
        }
      }
    };

    await linkRecursively(rootItems, null);
  }
}

module.exports = { handleNavigationEntry };
