"use strict";
const fs = require("fs-extra");
const path = require("path");
const mime = require("mime-types");

const UPLOAD_FOLDER = path.join(path.resolve(), "data", "uploads");

function getFileData(fileName) {
  const filePath = path.join(UPLOAD_FOLDER, fileName);

  if (!fs.existsSync(filePath)) {
    return null;
  }

  const stats = fs.statSync(filePath);
  const ext = path.extname(fileName).substring(1);

  return {
    filepath: filePath,
    originalFileName: fileName,
    size: stats.size,
    mimetype: mime.lookup(ext) || "application/octet-stream",
  };
}

function normalizeFileName(file) {
  if (typeof file === "string") {
    return file.split("/").pop();
  }
  if (file?.name) {
    const baseName = file.name.split("/").pop();
    const ext = file.ext ? file.ext : path.extname(baseName);
    return `${path.basename(baseName, ext)}${ext}`;
  }
  return null;
}

async function findExistingFile(fileName) {
  const parsed = path.parse(fileName);
  const baseName = parsed.name;
  const ext = parsed.ext;

  return await strapi.query("plugin::upload.file").findOne({
    where: {
      name: baseName,
      ext: ext || null,
    },
  });
}

async function uploadFileToStrapi(fileData, fileName) {
  const [uploadedFile] = await strapi
    .plugin("upload")
    .service("upload")
    .upload({
      files: fileData,
      data: {
        fileInfo: {
          name: fileName,
          alternativeText: fileData.alternativeText || "",
        },
      },
    });

  return uploadedFile;
}

async function processFile(fileName, alternativeText) {
  const existingFile = await findExistingFile(fileName);
  if (existingFile) {
    console.log(`[SKIP] File already exists in Strapi: ${fileName}`);
    return { success: true, file: existingFile, isNew: false };
  }

  const fileData = getFileData(fileName);
  if (!fileData) {
    return { success: false, error: `File not found: ${fileName}` };
  }

  try {
    const parsed = path.parse(fileName);
    const uploadedFile = await uploadFileToStrapi(
      { ...fileData, alternativeText },
      parsed.name
    );

    console.log(`[OK] Uploaded new file: ${fileName}`);
    return { success: true, file: uploadedFile, isNew: true };
  } catch (error) {
    console.error(`[ERROR] Upload failed for ${fileName}: ${error.message}`);
    return { success: false, error: error.message };
  }
}

async function checkFileExistsBeforeUpload(fileNames, alternativeText) {
  if (!Array.isArray(fileNames) || fileNames.length === 0) {
    return [];
  }

  const processedFiles = [];

  for (const file of fileNames) {
    const fileName = normalizeFileName(file);
    if (!fileName) continue;
    const result = await processFile(fileName, alternativeText);

    if (result.success) {
      processedFiles.push(result.file);
    }
  }

  return processedFiles.length === 1 ? processedFiles[0] : processedFiles;
}

module.exports = {
  checkFileExistsBeforeUpload,
  uploadFile: uploadFileToStrapi,
  getFileData,
};
